<?php

namespace App\Http\Resources\API\Order;

use App\Domain\Order\Actions\CanSelectServicesOrSparePartAction;
use App\Domain\Order\Actions\OrderCheckStatusAction;
use App\Domain\Order\Models\Order;
use App\Domain\Order\OrderStatus\Enums\AppointmentOrderStatusEnum;
use App\Domain\Order\OrderStatus\Enums\MarketPlaceOrderStatusEnum;
use App\Domain\Order\OrderStatus\Enums\SparePartOrderStatusEnum;
use App\Domain\Order\OrderStatus\Enums\WenchOrderStatusEnum;
use App\Http\Resources\API\Core\BranchResource;
use App\Http\Resources\API\Core\BrandResource;
use App\Http\Resources\API\Core\CarTypeResource;
use App\Http\Resources\API\Core\ProblemResource;
use App\Http\Resources\API\User\UserAddressesResource;
use App\Support\Traits\WithPagination;
use Carbon\Carbon;
use Illuminate\Http\Resources\Json\JsonResource;

class OrderDetailsResource extends JsonResource
{
    use WithPagination;

    /**
     * Transform the resource into an array.
     *
     * @param \Illuminate\Http\Request $request
     * @return array|\Illuminate\Contracts\Support\Arrayable|\JsonSerializable
     */
    public function toArray($request)
    {
        $status = match (true) {
            ($this?->type == 'appointment' && $this->status) =>
            AppointmentOrderStatusEnum::{$this->status}()->value,
            ($this?->type == 'spare_part' && $this->status) =>
            SparePartOrderStatusEnum::{$this->status}()->value,
            ($this?->type == 'wench' && $this->status) =>
            WenchOrderStatusEnum::{$this->status}()->value,
            ($this?->type == 'marketplace' && $this->status) =>
            MarketPlaceOrderStatusEnum::{$this->status}()->value,
            default => __('Status None')
        };
        $order = Order::find($this->id);
        $order_attributes = [
            [
                'label' => __('Total Order'),
                'value' => $this->sub_total + $this->discount_amount + $this->promo_discount_amount . ' ' . __('SAR')
            ],
        ];
        if ($this->discount_amount || $this->promo_discount_amount) {
            $order_attributes[] = [
                'label' => __('Discount Amount'),
                'value' => $this->discount_amount + $this->promo_discount_amount . ' ' . __('SAR')
            ];
            $order_attributes[] = [
                'label' => __('Total Order After discount'),
                'value' => $this->sub_total . ' ' . __('SAR')
            ];
        }
        $order_attributes[] =
            [
                'label' => __('Vat'),
                'value' => $this->vat_amount . ' ' . __('SAR')
            ];
        $order_attributes[] = [
            'label' => __('Final Amount'),
            'value' => $this->total_amount . ' ' . __('SAR')
        ];
        return [
            'id' => $this->id,
            'order_num' => $this->order_num,
            'status' => $this->status,
            'searching_for_driver' => $this->status === WenchOrderStatusEnum::pending()->label,
            'status_value' => $status,
            'qr_code' => $this->qr_code,
            'voucher' => new VoucherResource($this->voucher, $this->id),
            'qr_code_image' => $this->getFirstMediaUrl('qrCode'),
            'type' => $this->type,
            'type_for_customer' => __($this->type),
            'brand' => new BrandResource($this->brand),
            'car_type' => new CarTypeResource($this->car_type),
            'branch' => new BranchResource($this->branch),
            'problem' => new ProblemResource($this->problem),
            'payment_method' => $this->payment_method,
            'customer_note' => $this->customer_note,
            'delivery_direction' => $this->delivery_direction,
            'sub_total' => number_format($this->sub_total, 2, '.', ''),
            'delivery_amount' => number_format($this->delivery_amount, 2, '.', ''),
            'vat_amount' => number_format($this->vat_amount, 2, '.', ''),
            'discount_amount' => number_format($this->discount_amount, 2, '.', ''),
            'total_amount' => number_format($this->total_amount, 2, '.', ''),
            'price_before_discount' => number_format($this->sub_total + $this->discount_amount, 2, '.', ''),
            'old_paid_amount' => number_format($this->paidAmount(), 2, '.', ''),
            'amount_to_pay' => number_format($this->amountToPay(), 2, '.', ''),
            'wench_requests' => WenchRequestResource::collection($this->wenchRequests),
            'allow_scan_driver_code' => $this->status == WenchOrderStatusEnum::the_car_is_on_way_to_you()->label,
            'spare_parts' => SparePartOrderResource::collection($this->spareParts),
            'offers' => OfferOrderResource::collection($this->offers),
            'services' => ServiceOrderResource::collection($this->services),
            'address' => new UserAddressesResource($this->address),
            'reservation_time' => $this->reservation_time ? Carbon::parse($this->reservation_time)->translatedFormat('D, M j, Y g:i A') : null,
            'created_at' => $this->created_at->translatedFormat('D, M j, Y g:i A'),
            'can_pay' => $this->amountToPay() > 0 && (new CanSelectServicesOrSparePartAction())($this),
            'can_show_promo_code' => $this->amountToPay() > 0 && (new CanSelectServicesOrSparePartAction())($this),
            'can_cancel' => (new OrderCheckStatusAction())($order, 'canceled'),
            'view_invoice' => $this->amountToPay() == 0 && $this->status == AppointmentOrderStatusEnum::order_finished()->label,
            'open_cash' => $this->type != 'wench',
            'is_price_section_visible' => $this->amountToPay() > 0 && $this->status != AppointmentOrderStatusEnum::canceled()->label,
            'can_select_services' => (new CanSelectServicesOrSparePartAction())($this),
            'can_select_spare_part' => (new CanSelectServicesOrSparePartAction())($this),
            'allow_order_scanning' => $this->status == WenchOrderStatusEnum::truck_is_on_its_way_to_you()->label,
            'order_attributes' => $order_attributes
        ];
    }

}
